<?php

namespace Drupal\ey_common\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\file\Entity\File;
use Drupal\file\FileUsage\FileUsageInterface;

/**
 * Implements website configuration form.
 */
class SiteConfig extends ConfigFormBase {

  /**
   * The config factory object.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The file usage service.
   *
   * @var \Drupal\file\FileUsage\FileUsageInterface
   */
  protected $fileUsage;

  /**
   * Class constructor.
   */
  public function __construct(ConfigFactoryInterface $configFactory, FileUsageInterface $file_usage) {
    parent::__construct($configFactory);
    $this->fileUsage = $file_usage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('file.usage'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'ey_common.theme_settings',
    ];
  }

  /**
   * Returns the form’s unique ID.
   *
   * {@inheritdoc}.
   */
  public function getFormId() {
    return 'theme_settings_form';
  }

  /**
   * Returns the form array.
   *
   * {@inheritdoc}.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ey_common.theme_settings');
    
    $language = \Drupal::languageManager()->getCurrentLanguage()->getId();
    
    $form['election_status'] = [
      '#type' => 'radios',
      //'#title' => $this->t('Election Active'),
      '#options' => ['active' => 'Active election', 'inactive' => 'No active election'],
      '#default_value' => $config->get('election_status'),
    ];
    
    
    // Site Details Section
    $form['site_details'] = [
      '#type' => 'details',
      '#title' => $this->t('Site Details'),
      '#open' => TRUE,
    ];

    $form['site_details'][$language . '-site_name'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Site Name'),
      '#default_value' => $config->get($language . '-site_name'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Site Name')
      ),
    );

    $form['site_details'][$language . '-site_slogan'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Site Slogan'),
      '#default_value' => $config->get($language . '-site_slogan'),
    );

    $form['site_details'][$language . '-logo_upload'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Upload logo image'),
      '#upload_validators' => array(
          'file_validate_extensions' => array('png jpg jpeg'),
      ),
      '#description' => $this->t("If you don't have direct file access to the server, use this field to upload your logo."),
      '#theme' => 'image_widget',
      '#preview_imgage_style' => 'medium',
      '#upload_location' => 'public://site_details',
      '#progress_message' => 'One moment while we save your file...',
      '#default_value' => !empty($config->get('logo_upload')) ? $config->get($language . '-logo_upload') : '',
    ];

    $form['site_details'][$language . '-logo_alt_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Site Logo Alt Text'),
      '#default_value' => $config->get($language . '-logo_alt_text'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Logo Alt Text')
      ),
    ];

    $form['site_details'][$language . '-favicon_upload'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Upload favicon image'),
      '#upload_validators' => array(
          'file_validate_extensions' => array('ico png gif jpg jpeg apng svg'),
      ),
      '#description' => $this->t("If you don't have direct file access to the server, use this field to upload your shortcut icon."),
      '#theme' => 'image_widget',
      '#preview_imgage_style' => 'medium',
      '#upload_location' => 'public://site_details',
      '#progress_message' => 'One moment while we save your file...',
      '#default_value' => !empty($config->get($language . '-favicon_upload')) ? $config->get($language . '-favicon_upload') : '',
    ];

    // Social Links Section
    $form['social_links'] = [
      '#type' => 'details',
      '#title' => $this->t('Social Links'),
      '#open' => FALSE,
    ];

    $form['social_links'][$language . '-facebook_url'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Facebook Url'),
      '#default_value' => $config->get($language . '-facebook_url'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Facebook Url')
      ),
    );

    $form['social_links'][$language . '-twitter_url'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Twitter Url'),
      '#default_value' => $config->get($language . '-twitter_url'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Twitter Url')
      ),
    );

    $form['social_links'][$language . '-instagram_url'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Instagram Url'),
      '#default_value' => $config->get($language . '-instagram_url'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Instagram Url')
      ),
    );

    $form['social_links'][$language . '-bluesky_url'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Bluesky Url'),
      '#default_value' => $config->get($language . '-bluesky_url'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Bluesky Url')
      ),
    );

    // Footer Details Section
    $form['footer_details'] = [
      '#type' => 'details',
      '#title' => $this->t('Footer Details'),
      '#open' => FALSE,
    ];

    $form['footer_details'][$language . '-quick_links_title'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Section 1 Title'),
      '#default_value' => $config->get($language . '-quick_links_title'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Footer Section 1 Title')
      ),
    );

    $form['footer_details'][$language . '-contact_address_title'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Section 2 Title'),
      '#default_value' => $config->get($language . '-contact_address_title'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Footer Section 2 Title')
      ),
    );

    $form['footer_details'][$language . '-contact_address'] = array(
      '#type' => 'text_format',
      '#title' => $this->t('Contact Address'),
      '#default_value' => $config->get($language . '-contact_address')['value'],
      '#format' => 'full_html',
    );

    $form['footer_details'][$language . '-contact_details_title'] = array(
      '#type' => 'textfield',
      '#title' => $this->t('Section 3 Title'),
      '#default_value' => $config->get($language . '-contact_details_title'),
      '#attributes' => array(
        'placeholder' => $this->t('Enter Footer Section 3 Title')
      ),
    );

    $form['footer_details'][$language . '-contact_details'] = array(
      '#type' => 'text_format',
      '#title' => $this->t('Contact Details'),
      '#default_value' => $config->get($language . '-contact_details')['value'],
      '#format' => 'full_html',
    );

    $form['footer_details'][$language . '-copyright'] = array(
      '#type' => 'text_format',
      '#title' => $this->t('Copyright'),
      '#default_value' => $config->get($language . '-copyright')['value'],
      '#format' => 'full_html',
    );

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    $language = \Drupal::languageManager()->getCurrentLanguage()->getId();
    $config = $this->config('ey_common.theme_settings');

    // Save Site Logo and it's usage.
    $logo_upload = $form_state->getValue($language . '-logo_upload');
    if ($logo_upload != $config->get($language . '-logo_upload')) {
      $logo = File::load($logo_upload[0]);
      if (!empty($logo)) {
        $logo->setPermanent();
        $logo->save();
        $this->fileUsage->add($logo, 'site_logo', 'site_logo', \Drupal::currentUser()->id());
      }
    }

    // Save Site Logo and it's usage.
    $favicon_upload = $form_state->getValue($language . '-favicon_upload');
    if ($favicon_upload != $config->get($language . '-favicon_upload')) {
      $favicon = File::load($favicon_upload[0]);
      if (!empty($favicon)) {
        $favicon->setPermanent();
        $favicon->save();
        $this->fileUsage->add($favicon, 'site_favicon', 'site_favicon', \Drupal::currentUser()->id());
      }
    }
    
    // Save Configuration Values
    $this->config('ey_common.theme_settings')
      ->set($language . '-site_name', $form_state->getValue($language . '-site_name'))
      ->set($language . '-site_slogan', $form_state->getValue($language . '-site_slogan'))
      ->set($language . '-logo_upload', $form_state->getValue($language . '-logo_upload'))
      ->set($language . '-logo_alt_text', $form_state->getValue($language . '-logo_alt_text'))
      ->set($language . '-favicon_upload', $form_state->getValue($language . '-favicon_upload'))
      ->set($language . '-facebook_url', $form_state->getValue($language . '-facebook_url'))
      ->set($language . '-twitter_url', $form_state->getValue($language . '-twitter_url'))
      ->set($language . '-instagram_url', $form_state->getValue($language . '-instagram_url'))
      ->set($language . '-bluesky_url', $form_state->getValue($language . '-bluesky_url'))
      ->set($language . '-quick_links_title', $form_state->getValue($language . '-quick_links_title'))
      ->set($language . '-contact_address_title', $form_state->getValue($language . '-contact_address_title'))
      ->set($language . '-contact_address', $form_state->getValue($language . '-contact_address'))
      ->set($language . '-contact_details_title', $form_state->getValue($language . '-contact_details_title'))
      ->set($language . '-contact_details', $form_state->getValue($language . '-contact_details'))
      ->set($language . '-copyright', $form_state->getValue($language . '-copyright'))
      ->set('election_status', $form_state->getValue('election_status'))
      ->save();
  }

}